<?php
/**
 * ===================================================================================
 * Send Message Page (for Admins - Mobile Responsive)
 * ===================================================================================
 *
 * File: send_message.php
 * Location: /homework_portal/admin/send_message.php
 *
 * Purpose:
 * 1. Provides a responsive form for the Super Admin to send messages.
 * 2. Allows sending to a specific teacher or broadcasting to all teachers.
 * 3. Handles an optional file attachment with a mobile-friendly UI.
 */

require_once '../config.php';
require_once 'admin_helpers.php'; // Correctly include the helper file

// --- Security Check ---
check_permission('send_message');

$admin_id = $_SESSION['admin_id'];
$errors = [];
$success_message = '';

// --- Handle Form Submission ---
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['send_message'])) {
    $recipient = $_POST['recipient_teacher_id'] ?? 'all';
    $title = trim($_POST['title'] ?? '');
    $message_content = trim($_POST['message'] ?? '');
    
    // File Upload Logic
    $file_path_for_db = null;
    if (isset($_FILES['attachment']) && $_FILES['attachment']['error'] == UPLOAD_ERR_OK) {
        $upload_dir_physical = __DIR__ . '/../uploads/admin_messages/';
        if (!is_dir($upload_dir_physical)) {
            mkdir($upload_dir_physical, 0755, true);
        }
        
        $file_name = uniqid() . '_' . basename($_FILES['attachment']['name']);
        $target_file = $upload_dir_physical . $file_name;
        
        if (move_uploaded_file($_FILES['attachment']['tmp_name'], $target_file)) {
            $file_path_for_db = 'uploads/admin_messages/' . $file_name;
        } else {
            $errors[] = "Error uploading attachment.";
        }
    }

    if (empty($title)) $errors[] = "A title is required.";
    if (empty($message_content)) $errors[] = "A message is required.";

    if (empty($errors)) {
        try {
            $pdo->beginTransaction();

            // Note: sender_type is 'admin'
            $sql = "INSERT INTO teacher_messages (admin_id, teacher_id, title, message, file_path, sender_type) VALUES (?, ?, ?, ?, ?, 'admin')";
            $stmt = $pdo->prepare($sql);

            if ($recipient === 'all') {
                // Broadcast to all teachers
                $all_teachers = $pdo->query("SELECT id FROM teachers")->fetchAll(PDO::FETCH_COLUMN);
                foreach ($all_teachers as $teacher_id) {
                    $stmt->execute([$admin_id, $teacher_id, $title, $message_content, $file_path_for_db]);
                }
                $success_message = "Broadcast message sent to all teachers successfully!";
            } else {
                // Send to a specific teacher
                $teacher_id = (int)$recipient;
                $stmt->execute([$admin_id, $teacher_id, $title, $message_content, $file_path_for_db]);
                $success_message = "Message sent successfully!";
            }

            $pdo->commit();
        } catch (PDOException $e) {
            $pdo->rollBack();
            $errors[] = "Database error: Could not send message.";
        }
    }
}

// Fetch teachers for the dropdown and unread count
try {
    $teachers = $pdo->query("SELECT id, full_name FROM teachers ORDER BY full_name ASC")->fetchAll();
    $unread_count = $pdo->query("SELECT COUNT(id) FROM teacher_messages WHERE sender_type = 'teacher' AND is_read = 0")->fetchColumn();
} catch (PDOException $e) {
    $teachers = [];
    $unread_count = 0;
    $errors[] = "Could not fetch teacher list.";
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Send Message - Admin Portal</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@400;500;600;700&family=Baloo+2:wght@700&display=swap" rel="stylesheet">
    <style>
        body { font-family: 'Poppins', sans-serif; background-color: #f0f9ff; }
        .font-brand { font-family: 'Baloo 2', cursive; }
        .sidebar-link { display: flex; align-items: center; padding: 0.75rem 1rem; border-radius: 0.5rem; transition: background-color 0.2s, color 0.2s; }
        .sidebar-link:hover, .sidebar-link.active { background-color: #e0f2fe; color: #0284c7; }
        .notification-badge { display: inline-flex; align-items: center; justify-content: center; width: 22px; height: 22px; border-radius: 9999px; background-color: #ef4444; color: white; font-size: 0.75rem; font-weight: bold; }
    </style>
</head>
<body class="flex flex-col min-h-screen">
    <div class="relative min-h-screen md:flex">
        <!-- Mobile menu button -->
        <div class="md:hidden flex justify-between items-center bg-white p-4 shadow-md">
             <a href="dashboard.php">
                <h1 class="text-3xl font-bold text-red-600 font-sans">TTTEDDDEL</h1>
            </a>
            <button id="mobile-menu-button" class="text-gray-500 hover:text-blue-600 focus:outline-none">
                <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 6h16M4 12h16m-7 6h7"></path></svg>
            </button>
        </div>

        <!-- Sidebar Navigation -->
        <aside id="sidebar" class="w-64 bg-white shadow-md absolute inset-y-0 left-0 transform -translate-x-full md:relative md:translate-x-0 transition-transform duration-300 ease-in-out z-30">
            <div class="p-4 border-b text-center">
                <a href="dashboard.php">
                    <h1 class="text-4xl font-bold text-red-600 tracking-widest font-sans">TTTEDDDEL</h1>
                    <p class="text-xs font-semibold text-blue-800 mt-1">Teacher-Student Interactive System</p>
                    <p class="text-xs text-blue-800">by Ele Infocomm Technologies</p>
                </a>
            </div>
            <nav class="p-4 space-y-2">
                <a href="dashboard.php" class="sidebar-link"><span>Dashboard</span></a>
                 <a href="inbox.php" class="sidebar-link flex justify-between items-center">
                    <span>Inbox</span>
                    <?php if ($unread_count > 0): ?><span class="notification-badge"><?php echo $unread_count; ?></span><?php endif; ?>
                </a>
                <?php if (has_permission('manage_admins')): ?><a href="manage_admins.php" class="sidebar-link"><span>Manage Admins</span></a><?php endif; ?>
                <?php if (has_permission('manage_teachers')): ?><a href="manage_teachers.php" class="sidebar-link"><span>Manage Teachers</span></a><?php endif; ?>
                <?php if (has_permission('manage_classes')): ?><a href="manage_classes.php" class="sidebar-link"><span>Manage Classes</span></a><?php endif; ?>
                <?php if (has_permission('manage_subjects')): ?><a href="manage_subjects.php" class="sidebar-link"><span>Manage Subjects</span></a><?php endif; ?>
                <?php if (has_permission('assign_subjects')): ?><a href="assign_subjects.php" class="sidebar-link"><span>Assign Subjects</span></a><?php endif; ?>
                <?php if (has_permission('send_message')): ?><a href="send_message.php" class="sidebar-link active"><span>Send Message</span></a><?php endif; ?>
            </nav>
        </aside>

        <!-- Main Content Area -->
        <div class="flex-1 flex flex-col overflow-hidden">
            <header class="bg-white shadow-sm border-b hidden md:flex"><div class="flex items-center justify-end p-4 w-full"><span class="text-gray-600 mr-4">Welcome, <?php echo e($_SESSION['admin_username'] ?? 'Admin'); ?>!</span><a href="account_settings.php" class="p-2 text-gray-500 hover:text-blue-600" title="Account Settings"><svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M10.325 4.317c.426-1.756 2.924-1.756 3.35 0a1.724 1.724 0 002.573 1.066c1.543-.94 3.31.826 2.37 2.37a1.724 1.724 0 001.065 2.572c1.756.426 1.756 2.924 0 3.35a1.724 1.724 0 00-1.066 2.573c.94 1.543-.826 3.31-2.37 2.37a1.724 1.724 0 00-2.572 1.065c-.426 1.756-2.924 1.756-3.35 0a1.724 1.724 0 00-2.573-1.066c-1.543.94-3.31-.826-2.37-2.37a1.724 1.724 0 00-1.065-2.572c-1.756-.426-1.756-2.924 0-3.35a1.724 1.724 0 001.066-2.573c-.94-1.543.826-3.31 2.37-2.37.996.608 2.296.07 2.572-1.065z" /><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 12a3 3 0 11-6 0 3 3 0 016 0z" /></svg></a><a href="logout.php" class="ml-2 py-2 px-4 bg-blue-600 text-white font-semibold rounded-lg shadow-md hover:bg-blue-700">Logout</a></div></header>

            <main class="flex-1 overflow-x-hidden overflow-y-auto bg-gray-100 p-4 md:p-6">
                <h2 class="text-3xl font-semibold text-gray-800">Send Message</h2>
                
                <?php if ($success_message): ?><div class="p-4 my-4 text-sm text-green-800 bg-green-100 rounded-lg"><?php echo e($success_message); ?></div><?php endif; ?>
                <?php if (!empty($errors)): ?><div class="p-4 my-4 text-sm text-red-800 bg-red-100 rounded-lg"><ul class="list-disc list-inside"><?php foreach ($errors as $error) echo "<li>" . e($error) . "</li>"; ?></ul></div><?php endif; ?>

                <div class="mt-6 bg-white p-6 rounded-lg shadow-lg">
                    <form action="send_message.php" method="POST" enctype="multipart/form-data" class="space-y-6">
                        <div>
                            <label for="recipient_teacher_id" class="block text-sm font-medium text-gray-700">Recipient</label>
                            <select name="recipient_teacher_id" id="recipient_teacher_id" class="mt-1 block w-full p-2 border border-gray-300 rounded-md shadow-sm" required>
                                <option value="all">All Teachers (Broadcast)</option>
                                <?php foreach($teachers as $teacher): ?>
                                    <option value="<?php echo e($teacher['id']); ?>"><?php echo e($teacher['full_name']); ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div>
                            <label for="title" class="block text-sm font-medium text-gray-700">Title</label>
                            <input type="text" name="title" id="title" class="mt-1 block w-full p-2 border border-gray-300 rounded-md shadow-sm" required>
                        </div>
                        <div>
                            <label for="message" class="block text-sm font-medium text-gray-700">Message</label>
                            <textarea name="message" id="message" rows="8" class="mt-1 block w-full p-2 border border-gray-300 rounded-md shadow-sm" required></textarea>
                        </div>
                        <div>
                            <label for="attachment" class="block text-sm font-medium text-gray-700">Attachment (Optional)</label>
                            <input type="file" name="attachment" id="attachment" class="mt-1 block w-full text-sm text-gray-500 file:mr-4 file:py-2 file:px-4 file:rounded-full file:border-0 file:font-semibold file:bg-blue-50 file:text-blue-700 hover:file:bg-blue-100">
                        </div>
                        <div class="text-right border-t pt-6">
                            <button type="submit" name="send_message" class="w-full sm:w-auto px-8 py-3 bg-blue-600 text-white font-bold rounded-lg shadow hover:bg-blue-700">Send Message</button>
                        </div>
                    </form>
                </div>
            </main>
        </div>
    </div>
    <footer class="text-center py-6 bg-gray-100 border-t mt-auto">
        <p class="text-sm text-gray-500">Built By Ele InfoComm Technologies- Kenya | <a href="http://web.e-ict.co.ke" target="_blank" class="hover:underline">web.e-ict.co.ke</a> | <a href="mailto:info@e-ict.co.ke" class="hover:underline">info@e-ict.co.ke</a> | +254202000121 | +254 722 215810 | 2025 All right reserved.</p>
    </footer>

    <script>
        const btn = document.getElementById('mobile-menu-button');
        const sidebar = document.getElementById('sidebar');

        btn.addEventListener('click', () => {
            sidebar.classList.toggle('-translate-x-full');
        });
    </script>
</body>
</html>
